#include <pthread.h>
#include <assert.h>
#include <string.h>
#include "queue.h"
#include <stdio.h>

#define min(a,b) ((a)<(b) ? (a) : (b))

#if 0
#define dprintf(...) fprintf(stderr,__VA_ARGS__)
#else
#define dprintf(...) /* noop */
#endif

void queueInit(tQueue *q)
{
  pthread_mutex_init(&q->mutex,NULL);
  q->head = q->tail = q->count = 0;
  q->size = (int) sizeof q->data;
}

int queueCount(tQueue *q)
{
  int count;
  pthread_mutex_lock(&q->mutex);
  count = q->count;
  pthread_mutex_unlock(&q->mutex);
  return count;
}

int queueRoom(tQueue *q)
{
  return (q->size-1) - queueCount(q);
}

int queueEmpty(tQueue *q)
{
  return queueCount(q) == 0;
}

int queueFull(tQueue *q)
{
  return queueCount(q) ==  q->size-1;
}

void queueFlush(tQueue *q)
{
  pthread_mutex_lock(&q->mutex);
  q->head = q->tail = q->count = 0;
  pthread_mutex_unlock(&q->mutex);
}

#if defined(NDEBUG)
#define queueCheck(q) /* noop */
#else
#define queueCheck(q) \
do \
{ \
  int count; \
  assert (q->size > 0); \
  assert (q->count >= 0); \
  assert (q->count < q->size); \
  dprintf("h=%d t=%d c=%d\n",q->head,q->tail,q->count); \
  count = q->head - q->tail; \
  if (count<0) \
    count += q->size; \
  assert(count == q->count); \
} while(0)
#endif


void queueRewind(tQueue *q, unsigned count)
{
  int amount = (int)count;
  int room;
  pthread_mutex_lock(&q->mutex);
  queueCheck(q);
  room = (q->size-1) - q->count;
  amount = min(amount,q->size-1);
  amount = min(amount,q->wrcount);
  amount = min(amount,room);
  q->tail -= amount;
  if (q->tail < 0)
    q->tail += q->size;
  q->count += amount;
  queueCheck(q);
  pthread_mutex_unlock(&q->mutex);
}

int queueRead(tQueue *q, unsigned char *buf, unsigned count)
{
  int total;
  int requested = (int)count;
  
  pthread_mutex_lock(&q->mutex);
  dprintf("r locked\n");
  queueCheck(q);

  total = min(requested,q->count);
  
  dprintf("r requested=%d avail=%d total=%d\n",requested,q->count,total);
  
  if (total > q->size - q->tail)
    {
      // wrap
      int c1 = q->size - q->tail;
      int c2 = total - c1;
      dprintf("r c1=%d c2=%d\n",c1,c2);
      memcpy(buf,q->data+q->tail,c1);
      memcpy(buf+c1,q->data,c2);
    }
  else
    {
      memcpy(buf,q->data+q->tail,total);
    }
  
  q->count -= total;
  q->tail += total;
  if (q->tail >= q->size)
    q->tail -= q->size;

  queueCheck(q);
  
  pthread_mutex_unlock(&q->mutex);
  dprintf("r unlocked\n");
  return total;
}


int queueWrite(tQueue *q, unsigned char *buf, unsigned count)
{
  int requested = (int)count;
  int room,total;
  
  pthread_mutex_lock(&q->mutex);
  dprintf("w locked\n");
  
  queueCheck(q);
  
  room = (q->size - q->count) - 1;
  
  total = min(room,requested);

  dprintf("w req=%d room=%d total=%d\n",requested,room,total);
  
  if (total > q->size - q->head)
    {
      // copy is going to wrap
      int c1 = q->size - q->head;
      int c2 = total - c1;
      dprintf("w c1=%d c2=%d\n",c1,c2);
      memcpy(q->data+q->head,buf,c1);
      memcpy(q->data,buf+c1,c2);
    }
  else
    {
      // single copy
      memcpy(q->data+q->head,buf,total);
    }

  q->count += total;
  if (q->wrcount < q->size)
    q->wrcount += total;
  q->head += total;
  if (q->head >= q->size)
    q->head -= q->size;

  queueCheck(q);
  pthread_mutex_unlock(&q->mutex);
  dprintf("w unlocked\n");
  return total;
}

