#include <pthread.h>
#include <memory.h>
#include <stdio.h>
#include <fcntl.h>
#include <termios.h>
#include <unistd.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <time.h>
#include <signal.h>
#include <sys/time.h>
#include <sys/poll.h>
#include <assert.h>
#include <signal.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <stdarg.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <fcntl.h>
#if defined(ALT_NCURSES_INCLUDE)
#include <ncurses/panel.h>
#include <ncurses/form.h>
#else
#include <panel.h>
#include <form.h>
#endif
#include <ctype.h>

#include "getargs.h"
#include "queue.h"
#include "common.h"
#include "port.h"
#include "modes.h"
#include "version.h"

#define Ctrl(c) ((c) & 0x1f)
#define Meta(c) ((c) | 0x7f00)


#define err_abort(code,text) do {\
   endwin(); \
   fprintf(stderr,"\n%s at %s:%d: %s\n\n", \
	          text,__FILE__,__LINE__,strerror(code)); \
   abort(); \
   } while (0)

#define abort_if_error(s) do {if (s) err_abort(s,"");} while(0)

// variables set by command line options
static int verify;
static int quiet;
int debugLevel;
static int statusInBorder;
static int autoRaise;
static int hubmode;
typedef enum {ModeTerminal, ModeMonitor, ModeThroughput, ModeEcho, ModeTest} tPortMode;
static char *modeStrings[]={"term","mon","tput","echo","test",NULL};
static tPortMode defaultMode;
static int baudRate = 9600;
static int dataBits = 8;
static int stopBits = 1;
static enum { ParityNone, ParityEven, ParityOdd, ParityMark, ParitySpace } parity;
static char *parityStrings[] = { "none", "even", "odd", "mark", "space", NULL };
static enum { FlowNone, FlowHw, FlowSw} flowType;
static char *flowStrings[] = {"none","hw","sw",NULL};
static char *devName;
static int  schedFifo = 0;
static char *zeroCmd = NULL;
static char *errlogName = NULL;
static int  testTime = 0;
static int  averageTime = 1;
static int  refreshTime = 200;
static int  nonInteractive = 0;
static int  summaryReport = 0;

static int devNameHandler(char *device);

// command line option specs
static argSpec argSpecArray[] = 
{
    {'x', OptionInteger, &debugLevel, NULL, "debug level", NULL},
    {'B', OptionBoolean, &statusInBorder, NULL, "use top border for status line", NULL},
    {'a', OptionBoolean, &autoRaise, NULL, "auto-raise active window", NULL},
    {'q', OptionBoolean, &quiet, NULL, "quiet-mode (don't display raw data)", NULL},
    {'v', OptionBoolean, &verify, NULL, "verify receive data", NULL},
    {'h', OptionBoolean, &hubmode, NULL, "allow delays for serial hub", NULL},
    {'m', OptionEnumerated, &defaultMode, NULL, "default port mode", modeStrings},
    {'f', OptionEnumerated, &flowType, NULL, "flow control", flowStrings},
    {'w', OptionInteger, &dataBits, NULL, "word length (number of data bits)", NULL},
    {'s', OptionInteger, &stopBits, NULL, "stop bits", NULL},
    {'p', OptionEnumerated, &parity, NULL, "parity", parityStrings},
    {'b', OptionInteger, &baudRate, NULL, "baud rate", NULL},
    {'d', OptionString, &devName, devNameHandler, "device path", NULL},
    {'F', OptionBoolean, &schedFifo, NULL, "run comm threads with SCHED_FIFO policy", NULL},
    {'z', OptionString, &zeroCmd, NULL, "shell command to execute if throughput drops to zero", NULL},
    {'l', OptionString, &errlogName, NULL, "filename for error/status log", NULL},
    {'t', OptionInteger, &testTime, NULL, "how long to run (def 0 == forever)", NULL},
    {'A', OptionInteger, &averageTime, NULL, "period (sec) for throughput averaging (def == 1 sec)", NULL},
    {'r', OptionInteger, &refreshTime, NULL, "period (msec) for bytecount updates (def == 200 msec)", NULL},
    {'i', OptionBoolean, &nonInteractive, NULL, "non-interactive: no wait for 'Enter' when displaying messages", NULL},
    {'S', OptionBoolean, &summaryReport, NULL, "print summary throughtput report at termination", NULL},
};

static int argSpecCount = (sizeof argSpecArray / sizeof argSpecArray[0]);
static char *progName;
static void usage(void)
{
  fprintf(stderr,
	  "usage: %s [options] dev [dev [...]]\n"
	  "       to display options: %s -?\n", progName, progName);
}

FILE* errfp;

pthread_mutex_t log_mutex = PTHREAD_MUTEX_INITIALIZER;

typedef struct ChanData
{
  int x, y, w, h;
  char path[256];
  char label[256];
  int doLabelUpdate;
  PANEL *panel;
  WINDOW *win;
  WINDOW *statuswin;
  WINDOW *datawin;
  tPortData port;
  int oldx, oldy, oldrows, oldcols;
  int mode;
  int modemStatus;
  int quiet;
  struct ChanData *next;
}
tChanData;

static tChanData *chanList;
static tChanData *active;

// flag set when any active port's rxThroughput drops to 0.0
int zeroThroughputFlag;

// initial location for new windows
static int nextx = 0;
static int nexty = 0;

static void clipwin(int *rows, int *cols, int *y, int *x)
{
  int scrx, scry;
  getmaxyx(stdscr,scry,scrx);

  if (*y<0)
    *y = 0;
  if (*y >= scry-1)
    *y = scry-1;
  
  if (*x<0)
    *x = 0;
  if (*x >= scrx-1)
    *x = scrx-1;
  
  if (*x+*cols >= scrx)
    *cols = scrx - *x;
  if (*cols<3)
    *cols = 3;
  
  if (*rows+*y >= scry)
    *rows = scry - *y;
  if (*rows<4)
    *rows = 4;

  if (*x+*cols >= scrx && *cols == 3)
    *x = scrx-3;
  
  if (*rows+*y >= scry && *rows == 4)
    *y = scry-4;
}

static void trimncpy(char *dst, char *src, int max)
{
  int count;
  char *end;
  
  end = src + strlen(src) - 1;
  while (*end == ' ' && end>=src)
    --end;
  
  count = end-src + 1;

  if (count >= max)
    count = max-1;

  memcpy(dst,src,count);
  dst[count] = '\0';
}

static void displayMessage(char *format, ...)
{
  va_list ap;
  WINDOW *w,*s;
  PANEL *p;
  int c,x,y,rows,cols,maxy,maxx;
  char *prompt = "Press Enter";

  getmaxyx(stdscr,maxy,maxx);
  cols = 40;
  rows = 12;
  x = (maxx-cols)/2;
  y = (maxy-rows)/2;
  clipwin(&rows,&cols,&y,&x);
  
  w = newwin(rows,cols,y,x);                     assert(w);
  s = derwin(w,rows-4,cols-6,2,2);               assert(s);
  p = new_panel(w);                              assert(p);
  
  va_start(ap,format);
  vwprintw(s,format,ap);
  va_end(ap);
  wsyncup(s);
  mvwaddstr(w,rows-2,(cols-strlen(prompt))/2,prompt);
  box(w,0,0);

  cbreak();
  update_panels();
  doupdate();
  if (nonInteractive)
    sleep(1);
  else
    while ((c=getch()) != 0x0a)
      ;
  del_panel(p);
  delwin(s);
  delwin(s);
  halfdelay(1);
}

static void promptDialog(char *buf, int max, char *prompt, char *defval)
{
  FIELD *field[2];
  FORM  *f;
  WINDOW *w,*s;
  PANEL *p;
  int maxy,maxx,c;
  
  getmaxyx(stdscr,maxy,maxx);
  field[0] = new_field(1,(maxx/2)-4,0,0,0,0);     assert(field[0]);
  field[1] = NULL;
  
  set_field_back(field[0],A_UNDERLINE);
  field_opts_off(field[0],O_STATIC);
  if (defval)
    set_field_buffer(field[0],0,defval);
  
  f = new_form(field);                           assert(f);
  w = newwin(6,maxx/2,maxy/2-3,maxx/4);          assert(w);
  s = derwin(w,1,maxx/2-4,4,2);                  assert(s);
  
  set_form_win(f,w);
  set_form_sub(f,s);
  
  p = new_panel(w);                              assert(p);
  
  box(w,0,0);
  mvwaddstr(w,2,2,prompt);

  post_form(f);
  cbreak();
  keypad(w,1);
  curs_set(1);

  form_driver(f,REQ_END_FIELD);
  
  while ((c=wgetch(w)) != 0x0a)
    switch (c)
      {
       case Ctrl('a'):
        form_driver(f,REQ_BEG_FIELD);
        break;
       case Ctrl('e'):
        form_driver(f,REQ_END_FIELD);
        break;
       case Ctrl('k'):
        form_driver(f,REQ_CLR_EOF);
        break;
       case KEY_LEFT:
       case Ctrl('b'):
        form_driver(f,REQ_LEFT_CHAR);
        break;
       case KEY_RIGHT:
       case Ctrl('f'):
        form_driver(f,REQ_RIGHT_CHAR);
        break;
       case KEY_DC:
       case 0x7f:
       case Ctrl('d'):
        form_driver(f,REQ_DEL_CHAR);
        break;
       case KEY_BACKSPACE:
       case Ctrl('h'):
        form_driver(f,REQ_DEL_PREV);
        break;
       default:
        form_driver(f,c);
        break;
      }
  form_driver(f,REQ_VALIDATION);
  trimncpy(buf,field_buffer(field[0],0),max);
      
  curs_set(0);
  halfdelay(1);
  unpost_form(f);

  free_form(f);
  free_field(field[0]);
  del_panel(p);
  delwin(s);
  delwin(w);
}

static void updateLabel(tChanData *ch, char *label, int color)
{
  assert(ch->statuswin);
  wattron(ch->statuswin,color | A_BOLD);
  mvwaddstr(ch->statuswin,0,0,label);
  waddstr(ch->statuswin,"                                                                                                 ");
  wsyncup(ch->statuswin);
  strncpy(ch->label,label,sizeof ch->label);
  ch->doLabelUpdate = 0;
}

static void updateData(tChanData *ch, char *data, unsigned count)
{
  assert(ch); assert(ch->datawin); assert(data);
  while (count--)
    {
      if (*data != '\n')
        waddch(ch->datawin, *data);
      else
        {
          int y, x, maxy, maxx __attribute__((unused));
          getyx(ch->datawin,y,x);
          getmaxyx(ch->datawin,maxy,maxx);
          if (y < maxy-1)
            wmove(ch->datawin,y+1,x);
          else
            scroll(ch->datawin);
        }
      ++data;
    }
  wsyncup(ch->datawin);
}

static void moveWindow(tChanData *ch, int rows, int cols, int y, int x)
{
  WINDOW *win;
  clipwin(&rows, &cols, &y, &x);

  win = newwin(rows,cols,y,x);                     assert(win != NULL);

  Log(9,"moveWindow(%p, %d,%d, %d,%d)\n",ch,rows,cols,y,x);
      
  if (ch->statuswin) 
    delwin(ch->statuswin);
  if (ch->datawin) 
    delwin(ch->datawin);

  if (statusInBorder)
    {
      ch->statuswin = derwin(win,1,cols-2,0,1);     
      ch->datawin = derwin(win,rows-2,cols-2,1,1);
    }
  else
    {
      ch->statuswin = derwin(win,1,cols-2,1,1);
      ch->datawin = derwin(win,rows-3,cols-2,2,1);  
    }
  assert(ch->statuswin != NULL);
  assert(ch->datawin != NULL);
  
  getmaxyx(ch->datawin,rows,cols);
  queueRewind(&ch->port.monitorQueue,rows*cols);
  
  scrollok(ch->statuswin,0);
  wattron(ch->datawin,COLOR_PAIR(3) | A_BOLD);
  scrollok(ch->datawin,1);
  if (ch->panel)
    replace_panel(ch->panel,win);
  else
    ch->panel = new_panel(win);
  assert(ch->panel != NULL);
  
  if (ch->win)
    delwin(ch->win);
  
  ch->win = win;
  box(win, 0, 0);
  top_panel(ch->panel);
  updateLabel(ch,ch->path,COLOR_PAIR(1));
}

static void startTest(tChanData *chan)
{
  int s;
  sigset_t signalSet;
  pthread_attr_t attr;
  struct sched_param sp;
  
  tcflush(chan->port.fd,TCIOFLUSH);
  sigemptyset(&signalSet);
  sigaddset(&signalSet,SIGWINCH);
  s = pthread_sigmask(SIG_BLOCK, &signalSet, NULL);   abort_if_error(s);

  s = pthread_attr_init(&attr);  abort_if_error(s);
  if (schedFifo)
     {
	s = pthread_attr_setinheritsched(&attr, PTHREAD_EXPLICIT_SCHED); abort_if_error(s);
	s = pthread_attr_setschedpolicy(&attr, SCHED_FIFO); abort_if_error(s);
	s = pthread_attr_getschedparam(&attr, &sp); abort_if_error(s);
	sp.sched_priority = 50;
	s = pthread_attr_setschedparam(&attr, &sp); abort_if_error(s);
     }
   
  switch(chan->mode)
    {
     case ModeThroughput:
      s = pthread_create(&chan->port.threadId,&attr,modeThroughputThread,&chan->port);
      break;
     case ModeMonitor:
     case ModeTerminal:
      s = pthread_create(&chan->port.threadId,&attr,modeMonitorThread,&chan->port);
      break;
     case ModeEcho:
      s = pthread_create(&chan->port.threadId,&attr,modeEchoThread,&chan->port);
      break;
     case ModeTest:
      s = pthread_create(&chan->port.threadId,&attr,modeTestThread,&chan->port);
      break;
    }
  if (s)
     {
	endwin();
	err_abort(s,"creating thread");
     }
   
  s = pthread_sigmask(SIG_UNBLOCK, &signalSet, NULL);   abort_if_error(s);
}

static tChanData *openChan(char *device)
{
  struct termios t;
  tChanData *chan;
  int fd;
  char *p,*prefix="/dev/tty";
  
  if (!strchr(device,':'))
    {
      fd = open(device, O_RDWR | O_NOCTTY | O_NDELAY);
      if (fd >= 0)
        fcntl(fd, F_SETFL, 0);
    }
  else
    {
      struct sockaddr_in addr;
      int sock,status;
      struct hostent *hostent;
      char s[512],*p;
      
      strncpy(s,device,sizeof s);
      p = strchr(s,':');                               assert(p);
      *p++ = 0;
      hostent = gethostbyname(s);
      if (!hostent)
        {
          displayMessage("Error resolving hostname '%s'\n",s);
          return NULL;
        }
      sock = socket(PF_INET,SOCK_STREAM,0);    assert(sock>=0);
      addr.sin_family = AF_INET;
      addr.sin_port = htons(atoi(p));
      memcpy(&addr.sin_addr,hostent->h_addr,sizeof addr.sin_addr);
      status = connect(sock,(struct sockaddr*)&addr, sizeof addr);
      if (status)
        fd = -1;
      else
        fd = sock;
    }
     
  if (fd < 0)
    {
      displayMessage("Error opening '%s'\n\n%s",device,strerror(errno));
      return NULL;
    }
  
  chan = calloc(1, sizeof (tChanData));               assert(chan != NULL);
  queueInit(&chan->port.monitorQueue);
  chan->mode = defaultMode;
  chan->port.fd = fd;
  chan->quiet = quiet;
  chan->port.verifyData = verify;
  chan->port.isatty = isatty(chan->port.fd);
  chan->port.isahub = hubmode;

  if (chan->port.isatty)
    {
      static int baudErrorShown;
      static int dataBitsErrorShown;
      long baudSelect;
      ErrnoAbort( tcgetattr(chan->port.fd,&t) );
      cfmakeraw(&t);
      t.c_cflag = CLOCAL+CREAD;
      t.c_oflag = 0;
      t.c_iflag = INPCK;
      
      switch (baudRate)
        {
         case 300:  baudSelect = B300;   break;
         case 600:  baudSelect = B600;   break;
         case 1200: baudSelect = B1200;  break;
         case 2400: baudSelect = B2400;  break;
         case 4800: baudSelect = B4800;  break;
         case 9600: baudSelect = B9600;  break;
         case 19200: baudSelect = B19200; break;
         case 38400: baudSelect = B38400; break;
         case 57600: baudSelect = B57600; break;
         case 115200: baudSelect = B115200; break;
         case 230400: baudSelect = B230400; break;
         case 460800: baudSelect = B460800; break;
         default:
          if (!baudErrorShown)
            displayMessage("%ld is illegal baud rate.\nUsing 9600.\n",baudRate);
          baudSelect = B9600;
          baudErrorShown = 1;
          break;
        }
      cfsetospeed(&t,baudSelect);
      cfsetispeed(&t,baudSelect);
      
      switch (parity)
	{
         default:
         case ParityNone: break;
         case ParityEven:  t.c_cflag |= PARENB; break;
         case ParityOdd:   t.c_cflag |= PARENB+PARODD; break;
         case ParityMark:  t.c_cflag |= PARENB+PARODD+CMSPAR; break;
         case ParitySpace: t.c_cflag |= PARENB+CMSPAR; break;
	}

      switch (dataBits)
        {
         default:
          if (!dataBitsErrorShown)
            displayMessage("%d is illegal word length.\nUsing 8 data bits.\n",dataBits);
          dataBitsErrorShown = 1;
         case 8: t.c_cflag |= CS8; break;
         case 7: t.c_cflag |= CS7; break;
         case 6: t.c_cflag |= CS6; break;
         case 5: t.c_cflag |= CS5; break;
        }
      
      switch (stopBits)
        {
         case 2: t.c_cflag |= CSTOPB; 
          break;
         default: 
          displayMessage("%d is illegal number of stop bits.\nUsing 1.\n",stopBits);
         case 1:
          break;
        }
      
      if (flowType == FlowSw)
          t.c_iflag |= IXON+IXOFF;
      else if (flowType == FlowHw)
          t.c_cflag |= CRTSCTS;
      
      ErrnoAbort( tcsetattr(chan->port.fd,TCSANOW,&t) );
      tcflush(chan->port.fd,TCIOFLUSH);
      tcflush(chan->port.fd,TCIOFLUSH);
    }

  strncpy(chan->port.path,device,sizeof chan->port.path);
  
  // toss /dev/tty prefix
  p = device;
  while (*p == *prefix)
    {
      ++p;
      ++prefix;
    }
  strncpy(chan->path,p,sizeof chan->path);
  gettimeofday(&chan->port.timeLastAvg,NULL);  
  chan->port.timeLastRefresh = chan->port.timeLastAvg;
  return chan;
}

static tChanData *newChan(char *device)
{
  int maxx,maxy,x,y,rows,cols;
  tChanData *chan;
  
  chan = openChan(device);
  
  if (!chan)
    return NULL;
  
  // create window and panel
  
  getmaxyx(stdscr,maxy,maxx);

  x = nextx;
  y = nexty;
  nexty += statusInBorder ? 1 : 2;
  rows = maxy/2;
  cols = maxx;
  
  clipwin(&rows, &cols, &y, &x);
  
  moveWindow(chan,rows,cols,y,x);

  // create thread
  startTest(chan);
  
  return chan;
}

static void raiseWindow(void)
{
  top_panel(active->panel);
}

static void lowerWindow(void)
{
  bottom_panel(active->panel);
}

static void maximizeWindow(void)
{
  int y,x,rows,cols;
  getbegyx(active->win, active->oldy, active->oldx);
  getmaxyx(active->win, active->oldrows, active->oldcols);
  getbegyx(stdscr,y,x);
  getmaxyx(stdscr,rows,cols);
  moveWindow(active,rows,cols,y,x);
}

static void restoreWindow(void)
{
  if (active->oldrows && active->oldcols)
    moveWindow(active,active->oldrows,active->oldcols,active->oldy,active->oldx);
}

void userMoveWindow(void)
{
  tChanData *ch = active;
  int rows,cols,y,x;
  int c;
  cbreak();
  top_panel(ch->panel);
  while (1)
    {
      refresh();
      update_panels();
      doupdate();
      getbegyx(ch->win,y,x);
      getmaxyx(ch->win,rows,cols);
      c = getch();
      if (c == 0x0a)  // enter
        break;
      switch (c)
        {
         case KEY_UP:
          --y;
          break;
         case KEY_DOWN:
          ++y;
          break;
         case KEY_RIGHT:
          ++x;
          break;
         case KEY_LEFT:
          --x;
          break;
        }
      moveWindow(ch,rows,cols,y,x);
    }
  halfdelay(1);
}


void userSizeWindow(void)
{
  tChanData *ch = active;
  int rows,cols,y,x;
  int c;
  cbreak();
  top_panel(ch->panel);
  while (1)
    {
      refresh();
      update_panels();
      doupdate();
      getbegyx(ch->win,y,x);
      getmaxyx(ch->win,rows,cols);
      c = getch();
      if (c == 0x0a)  // enter
        break;
      switch (c)
        {
         case KEY_UP:
          --rows;
          break;
         case KEY_DOWN:
          ++rows;
          break;
         case KEY_RIGHT:
          ++cols;
          break;
         case KEY_LEFT:
          --cols;
          break;
        }
      moveWindow(ch,rows,cols,y,x);
    }
  halfdelay(1);
}

static void doRestartTest(tChanData *ch)
{
  pthread_cancel(ch->port.threadId);
  pthread_join(ch->port.threadId,NULL);
  tcflush(ch->port.fd,TCIOFLUSH);
  ch->port.txCount = 0;
  ch->port.rxCount = 0;
  ch->port.eof = 0;
  ch->port.werror = 0;
  ch->port.rerror = 0;
  ch->port.verror = 0;
  ch->port.vcount = 0;
  ch->port.suspend = 0;
  ch->port.rxThroughput = 0.0;
  ch->port.txThroughput = 0.0;
  ch->port.noRxData = 0;
  queueFlush(&ch->port.monitorQueue);
  werase(ch->datawin);
  wsyncup(ch->datawin);
  usleep(100000);
  gettimeofday(&ch->port.timeLastAvg,NULL);
  ch->port.timeLastRefresh = ch->port.timeLastAvg;
  startTest(ch);
}


static void restartTest(void)
{
  if (active)
    doRestartTest(active);
}

static void restartTestAll(void)
{
  tChanData *ch;
  for (ch=chanList; ch; ch=ch->next)
    doRestartTest(ch);
  zeroThroughputFlag = 0;
}

static void clipWindows(void)
{
  tChanData *ch;
  int x,y,rows,cols;

  for (ch = chanList; ch != NULL; ch = ch->next)
    {
      getbegyx(ch->win,y,x);
      getmaxyx(ch->win,rows,cols);
      clipwin(&rows,&cols,&y,&x);
      moveWindow(ch,rows,cols,y,x);
    }
}

static char *baudToString(speed_t s)
{
  switch (s)
    {
     case B300: return "300"; break;
     case B600: return "600"; break;
     case B1200: return "1200"; break;
     case B2400: return "2400"; break;
     case B4800: return "4800"; break;
     case B9600: return "9600"; break;
     case B19200: return "19200"; break;
     case B38400: return "38400"; break;
     case B57600: return "57600"; break;
     case B115200: return "115200"; break;
     case B230400: return "230400"; break;
     case B460800: return "460800"; break;
     default: return "???"; break;
    }
}

static speed_t stringToBaud(char *s)
{
  switch(atol(s))
    {
     case 300: return B300; break;
     case 600: return B600; break;
     case 1200: return B1200; break;
     case 2400: return B2400; break;
     case 4800: return B4800; break;
     case 9600: return B9600; break;
     case 19200: return B19200; break;
     case 38400: return B38400; break;
     case 57600: return B57600; break;
     case 115200: return B115200; break;
     case 230400: return B230400; break;
     case 460800: return B460800; break;
     default: return B9600; break;
    }
}

static char *parityToString(int c)
{
  if ((c & PARENB) == 0)
    return "none";
  if (c & CMSPAR)
    {
      if (c & PARODD)
        return "mark";
      else
        return "space";
    }
  else
    {
      if (c & PARODD)
        return "odd";
      else
        return "even";
    }
}

static int stringToParity(char *s)
{
  if (!strncmp("odd",s,3))
    return PARENB+PARODD;
  if (!strncmp("even",s,4))
    return PARENB;
  if (!strncmp("mark",s,4))
    return PARENB+PARODD+CMSPAR;
  if (!strncmp("space",s,5))
    return PARENB+CMSPAR;
  if (!strncmp("none",s,4))
    return 0;
  return 0;
}

static char *csizeToString(int c)
{
  switch (c)
    {
     case CS5: return "5"; break;
     case CS6: return "6"; break;
     case CS7: return "7"; break;
     case CS8: return "8"; break;
     default: return "?"; break;
    }
}

static int stringToCsize(char *s)
{
  switch (atoi(s))
    {
     case 5: return CS5; break;
     case 6: return CS6; break;
     case 7: return CS7; break;
     default:
     case 8: return CS8; break;
    }
}

static char *stopToString(int c)
{
  if (c)
    return "2";
  else
    return "1";
}

static int stringToStop(char *s)
{
  if (!strncmp(s,"2",1))
    return CSTOPB;
  else
    return 0;
}

static int stringToMode(char *s)
{
  int i;
  
  for (i=0; modeStrings[i]; ++i)
    if (!strncmp(s,modeStrings[i],strlen(modeStrings[i])))
      return i;
  return ModeTerminal;
}

static char *flowToString(struct termios *t)
{
  if (t->c_cflag & CRTSCTS)
    return "hw";
  if (t->c_iflag & IXON)
    return "sw";
  else
    return "none";
}

static void stringToFlow(struct termios *t, char *s)
{
  t->c_iflag &= ~(IXON+IXOFF);
  t->c_cflag &= ~CRTSCTS;
  if (!strncmp("hw",s,2))
    t->c_cflag |= CRTSCTS;
  if (!strncmp("sw",s,2))
    t->c_iflag |= IXON+IXOFF;    
}

static void configChannel(void)
{
  struct termios t;
  FIELD *field[32];
  FORM  *f;
  WINDOW *w,*s;
  PANEL *p;
  int maxy,maxx,rows,cols,x,y,c,i;
  char *baudValues[] = {"300","600","1200","2400","4800","9600","19200","38400","57600","115200","230400","460800",NULL};
  char *stopValues[] = {"1","2",NULL};
  char *dataValues[] = {"5","6","7","8",NULL};
  char *yesNoValues[] = {"No","Yes",NULL};
  char *na[] = {"n/a",NULL};

  if (!active)
    return;

  memset(&field,0,sizeof field);
  
  field[0] = new_field(1,8,3,15,0,0);  assert(field[0]);
  field[1] = new_field(1,8,4,15,0,0);  assert(field[1]);
  field[2] = new_field(1,8,5,15,0,0);  assert(field[2]);
  field[3] = new_field(1,8,6,15,0,0);  assert(field[3]);
  field[4] = new_field(1,8,7,15,0,0);  assert(field[4]);
  field[5] = new_field(1,8,9,15,0,0);  assert(field[5]);
  field[6] = new_field(1,8,11,15,0,0);  assert(field[6]);
  field[7] = new_field(1,8,12,15,0,0);  assert(field[7]);

  set_field_type(field[0],TYPE_ENUM,active->port.isatty ? baudValues : na,0,0);
  set_field_type(field[1],TYPE_ENUM,active->port.isatty ? parityStrings : na,0,0);
  set_field_type(field[2],TYPE_ENUM,active->port.isatty ? dataValues : na,0,0);
  set_field_type(field[3],TYPE_ENUM,active->port.isatty ? stopValues : na,0,0);
  set_field_type(field[4],TYPE_ENUM,active->port.isatty ? flowStrings : na,0,0);
  set_field_type(field[5],TYPE_ENUM,modeStrings,0,0);
  set_field_type(field[6],TYPE_ENUM,yesNoValues,0,0);
  set_field_type(field[7],TYPE_ENUM,yesNoValues,0,0);

  if (active->port.isatty)
    {
      tcgetattr(active->port.fd, &t);
      set_field_buffer(field[0],0,baudToString(cfgetispeed(&t)));
      set_field_buffer(field[1],0,parityToString(t.c_cflag & (PARENB+PARODD+CMSPAR)));
      set_field_buffer(field[2],0,csizeToString(t.c_cflag & CSIZE));
      set_field_buffer(field[3],0,stopToString(t.c_cflag & CSTOPB));
      set_field_buffer(field[4],0,flowToString(&t));
    }
  set_field_buffer(field[5],0,modeStrings[active->mode]);
  set_field_buffer(field[6],0,yesNoValues[active->quiet]);
  set_field_buffer(field[7],0,yesNoValues[active->port.verifyData]);
  
  for (i=0; field[i]; ++i)
    field_opts_off(field[i],O_EDIT+O_AUTOSKIP);
  
  f = new_form(field);                           assert(f);

  getmaxyx(stdscr,maxy,maxx);
  
  cols = 50;
  rows = 20;
  x = (maxx-cols)/2;
  y = (maxy-rows)/2;

  clipwin(&rows,&cols,&y,&x);
  
  w = newwin(rows,cols,y,x);                     assert(w);
  s = derwin(w,rows-2,cols-2,1,1);               assert(s);
  set_form_win(f,w);
  set_form_sub(f,s);
  p = new_panel(w);                              assert(p);
  box(w,0,0);
  post_form(f);

  mvwprintw(s, 1,8,"%s Configuration",active->port.path);

  mvwaddstr(s, 3,3,"      Baud:");  
  mvwaddstr(s, 4,3,"    Parity:");  
  mvwaddstr(s, 5,3," Data Bits:");  
  mvwaddstr(s, 6,3," Stop Bits:");  
  mvwaddstr(s, 7,3,"      Flow:");
  
  mvwaddstr(s, 9,3,"      Mode:");
  mvwaddstr(s,11,3,"     Quiet:");
  mvwaddstr(s,12,3,"    Verify:");
  
  mvwaddstr(s,16,2,"Press ENTER when finished.");
  
  cbreak();
  keypad(w,1);
  curs_set(1);

  form_driver(f,REQ_END_FIELD);
  
  while ((c=wgetch(w)) != 0x0a)
    switch (c)
      {
       case Ctrl('i'):
       case KEY_DOWN:
        form_driver(f,REQ_NEXT_FIELD);
        break;
       case KEY_UP:
        form_driver(f,REQ_PREV_FIELD);
        break;
       case Ctrl('a'):
        form_driver(f,REQ_BEG_FIELD);
        break;
       case Ctrl('e'):
        form_driver(f,REQ_END_FIELD);
        break;
       case Ctrl('k'):
        form_driver(f,REQ_CLR_EOF);
        break;
       case KEY_LEFT:
       case Ctrl('b'):
        form_driver(f,REQ_LEFT_CHAR);
        break;
       case KEY_RIGHT:
       case Ctrl('f'):
        form_driver(f,REQ_RIGHT_CHAR);
        break;
       case KEY_DC:
       case 0x7f:
       case Ctrl('d'):
        form_driver(f,REQ_DEL_CHAR);
        break;
       case KEY_BACKSPACE:
       case Ctrl('h'):
        form_driver(f,REQ_DEL_PREV);
        break;
       case KEY_NPAGE:
        form_driver(f,REQ_PREV_CHOICE);
        break;
       case KEY_PPAGE:
       case (' '):
        form_driver(f,REQ_NEXT_CHOICE);
        break;
       default:
        form_driver(f,c);
        break;
      }
  form_driver(f,REQ_VALIDATION);

  if (active->port.isatty)
    {
      cfsetispeed(&t,stringToBaud(field_buffer(field[0],0)));
      cfsetospeed(&t,stringToBaud(field_buffer(field[0],0)));
      t.c_cflag = (t.c_cflag & ~(PARODD+PARENB+CMSPAR)) | stringToParity(field_buffer(field[1],0));
      t.c_cflag = (t.c_cflag & ~CSIZE) | stringToCsize(field_buffer(field[2],0));
      t.c_cflag = (t.c_cflag & ~CSTOPB) | stringToStop(field_buffer(field[3],0));
      stringToFlow(&t,field_buffer(field[4],0));
      tcsetattr(active->port.fd,TCSANOW,&t);
    }
  active->mode = stringToMode(field_buffer(field[5],0));
  active->quiet = !strncasecmp("yes",field_buffer(field[6],0),3);
  active->port.verifyData = !strncasecmp("yes",field_buffer(field[7],0),3);
  
  restartTest();
  
  curs_set(0);
  halfdelay(1);
  unpost_form(f);

  free_form(f);
  for (i=0; field[i]; ++i)
    free_field(field[i]);
  del_panel(p);
  delwin(w);
}

static void stackWindows(void)
{
  tChanData *ch;
  int v,x,y,rows,cols,maxx,maxy;

  getmaxyx(stdscr,maxy,maxx);
  x = y = 0;
  rows = maxy/2;
  cols = maxx;
  v = statusInBorder ? 1 : 2;
  
  for (ch = chanList; ch != NULL; ch = ch->next)
    {
      moveWindow(ch,rows,cols,y,x);
      y += v;
    }
}

static void cascadeWindows(void)
{
  tChanData *ch;
  int n,v,x,y,rows,cols,maxx,maxy;

  n=0;
  for (ch=chanList; ch!=NULL; ch=ch->next)
    ++n;
  
  getmaxyx(stdscr,maxy,maxx);
  x = y = 0;
  rows = maxy/2;
  cols = maxx-n*2;
  v = statusInBorder ? 1 : 2;
  
  for (ch = chanList; ch != NULL; ch = ch->next)
    {
      moveWindow(ch,rows,cols,y,x);
      y += v;
      x += 2;
    }
}

static void tileWindows(void)
{
  int n,r,c,maxy,maxx,rows,cols,i,j;
  tChanData *ch;

  if (chanList == NULL) return;
  
  n = 0;
  for (ch=chanList; ch; ch=ch->next)
    ++n;

  if (n==0)
    return;
  
  getmaxyx(stdscr,maxy,maxx);
  
  cols = maxx/40;
  rows = n/cols;
  if (rows*cols < n)
    ++rows;

  r = maxy/rows;
  c = maxx/cols;
  
  ch = chanList;
  for (i=0; i<rows && ch; ++i)
    for (j=0; j<cols && ch; ++j)
      {
        moveWindow(ch,r,c,i*r,j*c);
        ch = ch->next;
      }
}


static void nextWindow(void)
{
  if (!active) return;
  active->doLabelUpdate = 1;
  active = active->next;
  if (active == NULL)
    active = chanList;
  active->doLabelUpdate = 1;
  raiseWindow();
}

void noopCommand(void)
{
}

void openWindow(void)
{
  tChanData *ch, *p;
  char path[128];
  promptDialog(path,sizeof path, "Enter device name:","/dev/tty");
  nextx = nexty = 0;
  ch = newChan(path);
  if (ch == NULL)
    return;
  if (chanList == NULL)
    chanList = ch;
  else
    {
      // add to end of list
      for (p=chanList; ;p=p->next)
        if (p->next == NULL)
          {
            p->next = ch;
            break;
          }
    }
  active = ch;
}

void closeWindow(void)
{
  tChanData *cp, *a=active;
  if (!active) return;

  pthread_cancel(active->port.threadId);
  if (a->panel)  del_panel(a->panel);
  if (a->statuswin) delwin(a->statuswin);
  if (a->datawin) delwin(a->datawin);
  if (a->win) delwin(a->win);
  pthread_join(a->port.threadId,NULL);
  Log(1,"flush/close port %s\n",a->port.path);
  tcflush(a->port.fd,TCIOFLUSH);
  usleep(20000);
  tcflush(a->port.fd,TCIOFLUSH);
  close(a->port.fd);
  
  if (chanList == a)
    chanList = a->next;
  else
    for (cp=chanList; cp; cp = cp->next)
      if (cp->next == a)
        {
          cp->next = a->next;
          break;
        }
  active = a->next;
  if (active == NULL)
    active = chanList;
  if (active)
    active->doLabelUpdate = 1;
  free(a);
}

static void toggleQuiet(void)
{
  if (active)
    active->quiet = !active->quiet;
}

static void toggleQuietAll(void)
{
  tChanData *ch;
  for (ch=chanList; ch; ch=ch->next)
    ch->quiet = !ch->quiet;
}

static int sigwinchReceived;

static void sigwinchHandler(int sig)
{
  (void) sig;
  sigwinchReceived = 1;
}

void setModemStatus(int fd, int status)
{
  if (fd >= 0)
    ioctl(fd, TIOCMSET, &status);
}

int getModemStatus(int fd)
{
  int s,modemStatus;
  if (fd <= 0)
    return 0;
  s = ioctl(fd, TIOCMGET, &modemStatus);
  if (s)
    return 0;
  else
    return modemStatus;
}

static void showKeymap(void);

static void pauseResumeTest(void)
{
  if (active)
    active->port.suspend = !active->port.suspend;
}

static void toggleDTR(void)
{
  if (active && active->port.isatty)
    setModemStatus(active->port.fd, active->modemStatus ^ TIOCM_DTR);
}

static void toggleRTS(void)
{
  if (active && active->port.isatty)
    setModemStatus(active->port.fd, active->modemStatus ^ TIOCM_RTS);
}

static struct 
{
  int keyCode;
  void (*func)(void);
  char *helpString;
}commandTable[] =
{
    {Ctrl('I'), nextWindow, "Select next window as active"},
    {Meta('s'), stackWindows, "Stack windows"},
    {Meta('t'), tileWindows, "Tile windows"},
    {Meta('w'), cascadeWindows, "Waterfall windows"},
    {Meta('c'), closeWindow, "Close port and window"},
    {Meta('o'), openWindow, "Open new port"},
    {Meta('r'), raiseWindow, "Raise window"},
    {Meta('l'), lowerWindow, "Lower window"},
    {Meta('m'), maximizeWindow, "Maximize window"},
    {Meta('n'), restoreWindow,  "Restore window"},
    {Meta('p'), pauseResumeTest, "Pause (or resume) channel"},
    {Meta('M'), userMoveWindow, "Move window"},
    {Meta('S'), userSizeWindow, "Size window"},
    {Meta('D'), toggleDTR, "Toggle DTR"},
    {Meta('R'), toggleRTS, "Toggle RTS"},
    {Meta('h'), showKeymap, "Show help screen"},
    {Meta('i'), restartTest, "Reinitialize channel"},
    {Meta('I'), restartTestAll, "Reinitialize channel (all)"},
    {Meta('q'), toggleQuiet, "Toggle Quiet mode"},
    {Meta('Q'), toggleQuietAll, "Toggle Quiet mode (all)"},
    {Meta('e'), configChannel, "Edit port configuration"},
    {Meta('O'), NULL, "Send Xon (ctrl-Q) in terminal mode"},
    {Meta('F'), NULL, "Send Xoff (ctrl-S) in terminal mode"},
    {Meta('x'), NULL, "Exit program"},  // just for help screen
    {Meta('X'), NULL, "Exit program"},  // just for help screen
    {ERR,       NULL, NULL}  // ignore timeouts
};

void showKeymap(void)
{
  WINDOW *w,*t,*s;
  PANEL *p;
  int rows,cols,i;
  char *helpLines[128];
  char buf[1024];
  int offset;
  int trows,tcols;
  char *headerString = "HELP -- Up/Down to scroll -- Enter to exit";
  
  getmaxyx(stdscr,rows,cols);
  
  trows = rows-3;
  tcols = cols-2;
  
  w = newwin(rows,cols,0,0);           assert(w != NULL);
  p = new_panel(w);
  s = derwin(w,1,tcols,1,1);       assert(s != NULL);
  t = derwin(w,trows,tcols,2,1);  assert(t != NULL);
  
  scrollok(t,1);

  box(w,0,0);
  wbkgdset(s,A_REVERSE);
  mvwaddstr(s,0,(tcols-strlen(headerString))/2,headerString);
  wsyncup(s);
  
  memset(helpLines,0,sizeof helpLines);

  offset = 0;
  
  for (i=0; i<NumElements(commandTable); ++i)
    {
      char *modifier="";
      char key = ' ';
      if (offset >= NumElements(helpLines))
        break;
      if (commandTable[i].keyCode == ERR)
        continue;
      else if ((commandTable[i].keyCode & 0xff00) == Meta(0))
        {
          modifier = "Escape-";
          key = commandTable[i].keyCode & 0x7f;
        }
      else if (commandTable[i].keyCode < 0x20)
        {
          modifier = "Control-";
          key = commandTable[i].keyCode | 0x40;
        }
      sprintf(buf,"%10s%c:  %s",modifier,key,commandTable[i].helpString);
      helpLines[offset++] = strdup(buf);
      if (offset >= NumElements(helpLines))
        --offset;
    }
  
  offset = 0;

  for (i=offset; i<rows-1; ++i)
    {
      if (i>=NumElements(helpLines))
        break;
      mvwaddstr(t,i,0,helpLines[i]);
    }
  wsyncup(t);
  
  cbreak();
    
  while (1)
    {
      int c;
      refresh();
      update_panels();
      doupdate();
      c = getch();
      if (c == 0x0a) // enter
        break;
      else if (c == KEY_UP)
        {
          if (offset == 0)
            {
              beep();
              continue;
            }
          --offset;
          wscrl(t,-1);
          mvwaddstr(t,0,0,helpLines[offset]);
          wsyncup(t);
        }
      else if (c == KEY_DOWN)
        {
          if ((offset+trows-1) >= NumElements(helpLines))
            {
              beep();
              continue;
            }
          ++offset;
          wscrl(t,1);
          mvwaddstr(t,trows-1,0,helpLines[offset+trows-1]);
          wsyncup(t);
        }
      else
        beep();
    }

  del_panel(p);
  delwin(t);
  delwin(s);
  delwin(w);
  for (i=0; i<NumElements(helpLines);++i)
    if (helpLines[i])
      free(helpLines[i]);
  
  halfdelay(1);
}


static int devNameHandler(char *device)
{
  tChanData *chan;
  chan = newChan(device);
  if (chan)
    {
      chan->next = NULL;
      if (chanList == NULL)
        chanList = chan;
      else
        {
          tChanData *p;
          for (p=chanList; p->next; p=p->next)
            ;
          p->next = chan;
        }
    }
  return 0;
}

static void centerLine(WINDOW *w, int row, char *fmt, ...)
{
  char buf[512];
  va_list ap;
  int x,y __attribute__((unused)),c;
  
  getmaxyx(w,y,x);
  va_start(ap,fmt);
  vsnprintf(buf, sizeof buf, fmt, ap);
  va_end(ap);
  c = (x-strlen(buf))/2;
  mvwaddstr(w,row,c,buf);
}

static void showVersion(void)
{
  centerLine(stdscr,2,"lcom version %s",VersionString);
  centerLine(stdscr,4,"Copyright Comtrol Corp.");
  centerLine(stdscr,6,"Escape-h for help");
  centerLine(stdscr,7,"Escape-x to exit");
}

long timeDeltaMS(struct timeval *tNow, struct timeval *tLast)
{
  long deltaSeconds = tNow->tv_sec - tLast->tv_sec;
  long deltaMicroSeconds = tNow->tv_usec - tLast->tv_usec;
  return (deltaSeconds * 1000L) + deltaMicroSeconds/1000;
}

int main(int argc, char **argv)
{
  int i,c,s,argsUsed;
  tChanData *ch,*list=NULL;
  struct sigaction sigact;
  sigset_t signalSet;
  time_t endTime;

  progName = argv[0];

  argsUsed = getargs(argc, argv, argSpecArray, argSpecCount, 0);
  if (argsUsed < 0)
    {
      usage();
      exit(1);
    }
    
  argc -= argsUsed;
  argv += argsUsed;
  
  if (errlogName)
    {
      if (!(errfp = fopen(errlogName,"w")))
        {
          perror("opening error log file");
          exit(1);
        }
    }
  else
    errfp = stderr;
  
  setlinebuf(errfp);

  Log(1,"lcom version %s\n",VersionString);
  Log(1,"Copyright Comtrol Corp.\n");
  
  /* Initialize curses */
  initscr();
  start_color();
  cbreak();
  noecho();
  keypad(stdscr, TRUE);
  curs_set(0);
  
  Log(3,"Curses initialized\n");
  
  showVersion();
  
  use_default_colors();
  init_pair(1, COLOR_CYAN, COLOR_RED);
  init_pair(2, COLOR_WHITE, COLOR_RED);
  init_pair(3, COLOR_BLUE, -1);

  sigact.sa_flags = 0;
  sigact.sa_handler = sigwinchHandler;
  s = sigaction(SIGWINCH, &sigact, NULL);  assert(s==0);
  
  sigemptyset(&signalSet);
  sigaddset(&signalSet,SIGPIPE);
  s = pthread_sigmask(SIG_BLOCK, &signalSet, NULL);   abort_if_error(s);

  /* open ports */
  for (i=0; i<argc; ++i)
    {
      tChanData *chan;
      // check for port-range
      if (strchr(argv[i],'-'))
        {
          char *p;
          int n;
          int endport = -1;
          int startport = -1;
          char *s = strdup(argv[i]);

          p = s + strlen(s)-1;
          while ((*p != '-') && (p >= s))
            --p;
          endport = atoi(p+1);
          *p = '\0';
          --p;
          while (isdigit(*p))
            --p;
          ++p;
          startport = atoi(p);
          if (startport >= 0 && endport >= startport)
            for (n=startport; n<=endport; ++n)
              {
                sprintf(p,"%d",n);
                chan = newChan(s);
                if (chan)
                  {
                    chan->next = list;
                    list = chan;
                  }
              }
          free(s);
        }
      else
        {
          chan = newChan(argv[i]);
          if (chan)
            {
              chan->next = list;
              list = chan;
            }
        }
    }
  
  Log(2,"All ports opened\n");

  /* reverse temporary list */
  while (list)
    {
      tChanData *next = list->next;
      list->next = chanList;
      chanList = list;
      list = next;
    }
    
  active = chanList;
  doupdate();

  endTime = time(NULL) + testTime;
  
  halfdelay(1);
  
  while (testTime==0 || (time(NULL) < endTime))
    {
      c = getch();

      if (c == 0x1b)
        c = 0x7f00 | getch();

      if (sigwinchReceived)
        {
          struct winsize size;
          if (ioctl(fileno(stdout), TIOCGWINSZ, &size) == 0) 
            {
              resizeterm(size.ws_row, size.ws_col);
              clipWindows();
              Log(4,"SIGWINCH received: %dx%d\n",size.ws_row,size.ws_col);
            }
          sigwinchReceived = 0;
        }

      // getch() returns ERR for input key timeout
      if (c != ERR)
        {
          if (c == Meta('x') || c == Meta('X'))
            break;

          int commandFound = 0;
      
          for (i=0; i<NumElements(commandTable); ++i)
            {
              if (c == commandTable[i].keyCode)
                {
                  if (commandTable[i].func)
                    {
                      commandTable[i].func();
                      commandFound = 1;
                    }
                  break;
                }
            }

          if (!commandFound)
            {
              if (active && active->mode == ModeTerminal)
                {
                  if (c == Meta('O'))
                    c = Ctrl('Q');
                  else if (c == Meta('F'))
                    c = Ctrl('S');

                  if (((c & 0x7f) == c) ||   // normal ASCII
                      (c == Meta(0x1b)) ||   // ESC-ESC
                      (c == Meta(0x09)))     // ESC-TAB
                    {
                      // We're in terminal mode, so write byte to port
                      char b = c & 0x7f;
                      if (active->port.fd > 0)
                        {
                          if (write(active->port.fd,&b,1) == 1)
                            ++active->port.txCount;
                          else
                            {
                              displayMessage("Error writing to '%s'\n\n%s\n\nClosing port...",
                                             active->path,
                                             strerror(errno));
                              closeWindow();
                            }
                        }
                    }
                  else
                    {
                      Log(1,"command not found c = 0x%x\n",c);
                      beep();
                    }
                }
              else
                {
                  Log(1,"command not found c = 0x%x\n",c);
                  beep();
                }
            }
        }
      
      for (ch = chanList; ch != NULL; ch = ch->next)
        {
          int n;
          char buf[1024];
          long delta;
          struct timeval timeNow;

          gettimeofday(&timeNow,NULL);
          delta = timeDeltaMS(&timeNow,&ch->port.timeLastAvg);
          if (delta > averageTime*1000)
            {
              float d = (float)delta / 1e3;
              unsigned long rxCount = ch->port.rxCount;
              unsigned long txCount = ch->port.txCount;
              ch->port.rxThroughput = ((float)(rxCount - ch->port.rxCountLast))/d;
              ch->port.txThroughput = ((float)(txCount - ch->port.txCountLast))/d;
              ch->port.rxCountLast = rxCount;
              ch->port.txCountLast = txCount;
              if (ch->port.rxThroughput == 0.0)
                ch->port.noRxData = 1;
              ch->port.timeLastAvg = timeNow;
            }

          delta = timeDeltaMS(&timeNow,&ch->port.timeLastRefresh);
          if (delta > refreshTime)
            {
              if (ch->port.isatty)
                {
                  ch->modemStatus = getModemStatus(ch->port.fd);
                  sprintf(buf,"%s [%s] %s%s%s%s%s%s %s %s %s %s %s %s %lu/%lu %0.1f/%0.1f",
                          ch->path,
                          modeStrings[ch->mode],
                          ch->quiet ? "Q" : "",
                          ch->port.suspend ? "P" : "",
                          ch->port.verror ? "V" : "",
                          ch->port.rerror ? "R" : "",
                          ch->port.werror ? "W" : "",
                          ch->port.eof ? "E" : "",
                          ch->modemStatus & TIOCM_CD  ? "CD" : "cd",
                          ch->modemStatus & TIOCM_DTR ? "DTR" : "dtr",
                          ch->modemStatus & TIOCM_DSR ? "DSR" : "dsr",
                          ch->modemStatus & TIOCM_RI ?  "RI" : "ri",
                          ch->modemStatus & TIOCM_RTS ? "RTS" : "rts",
                          ch->modemStatus & TIOCM_CTS ? "CTS" : "cts",
                          ch->port.txCount,ch->port.rxCount,
                          ch->port.txThroughput,ch->port.rxThroughput);
                }
              else
                {
                  sprintf(buf,"%s [%s] %s%s%s%s%s%s %lu/%lu %0.1f/%0.1f",
                          ch->path,
                          modeStrings[ch->mode],
                          ch->quiet ? "Q" : "",
                          ch->port.suspend ? "P" : "",
                          ch->port.verror ? "V" : "",
                          ch->port.rerror ? "R" : "",
                          ch->port.werror ? "W" : "",
                          ch->port.eof ? "E" : "",
                          ch->port.txCount,ch->port.rxCount,
                          ch->port.txThroughput,ch->port.rxThroughput);
                }
              if (ch->doLabelUpdate || strcmp(ch->label,buf))
                updateLabel(ch,buf,active==ch ? COLOR_PAIR(2):COLOR_PAIR(1));
              ch->port.timeLastRefresh = timeNow;
            }
          if (!ch->quiet)
            {
              n = queueRead(&ch->port.monitorQueue, (unsigned char*)buf, sizeof buf);
              if (n)
                updateData(ch,buf,n);
            }
          if (!zeroThroughputFlag && ch->port.noRxData)
            {
              Log(4,"Zero throughput detected on port %s\n",ch->port.path);
              zeroThroughputFlag = 1;
              if (zeroCmd)
                {
                  Log(3,"Executing zero-throughtput command: '%s'\n",zeroCmd);
                  if (system(zeroCmd) == -1)
                    Log(3,"Zero-throughput command failed: '%s'\n",strerror(errno));
                }
            }
        }
      refresh();
      update_panels();
      doupdate();
    }



  // prepare summary report for printing later (data goes away when
  // windows are close, but we don't want to print it until after
  // curses is shut down.
  char summaryReportData[1024] = "";
  if (summaryReport)
    {
      char *p = summaryReportData;
      for (ch = chanList; ch != NULL; ch = ch->next)
        p += sprintf(p,"%0.1ff/%0.1f ",ch->port.txThroughput,ch->port.rxThroughput);
      // overwrite final space character with newline
      summaryReportData[strlen(summaryReportData)-1] = '\n';
    }

  while (active)
    closeWindow();
  Log(1,"Shutting down curses\n");
  endwin();
  Log(1,"Done\n");

  if (summaryReport)
    fprintf(stderr,"%s",summaryReportData);
  return 0;
}

