#include <pthread.h>
#include <stdio.h>
#include <fcntl.h>
#include <termios.h>
#include <unistd.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <time.h>
#include <signal.h>
#include <sys/time.h>
#include <sys/poll.h>
#include <sys/ioctl.h>
#include <stdarg.h>

#include "queue.h"
#include "common.h"
#include "port.h"
#include "modes.h"

#define max(a,b) (((a)>(b)) ? (a) : (b))

#if 0
static void dumpData(char *s, unsigned char *data, int count)
{
  int i;
  fprintf(stderr,"%s [%3d] ", s, count);
  if (count > 0)
    for (i = 0; i < count; ++i)
      fprintf(stderr,"%02x ", data[i]);
  fprintf(stderr,"\n");
}
#endif

static int qprintf(tQueue *q, char *fmt, ...)
{
  int  n;
  va_list ap;
  char buf[2048];
  va_start(ap,fmt);
  n = vsnprintf(buf,sizeof buf, fmt, ap);
  va_end(ap);
  return queueWrite(q,(unsigned char*)buf,n);
}


void nsleep(time_t seconds, long nanoseconds)
{
  struct timespec req = {seconds,nanoseconds};
  struct timespec rem;
  if (nanosleep(&req,&rem)<0)
    {
      if (errno == EINTR)
        nanosleep(&rem,NULL);
      else
        {
          char buf[256];
          strerror_r(errno,buf,sizeof buf);
          Log(1,"nanosleep error %d '%s'\n",errno,buf);
        }
    }
}

static unsigned char testPattern[256*8];

static pthread_once_t once_block = PTHREAD_ONCE_INIT;

static void modeInit(void)
{
  unsigned i;
  for (i=0; i<(sizeof testPattern); ++i)
    testPattern[i] = ' '+(i%(126-' '));
  setlinebuf(stderr);
}

void *modeTestThread(void * param)
{
  int count,i;
  unsigned char buf[sizeof testPattern];
  int modemStatus;
  struct pollfd pfd;
  long delta;
  int dtrCdTestFail=0, dtrDsrTestFail=0;
  int rtsRiTestFail=0, rtsCtsTestFail=0;
  tPortData *port = param;
  tQueue *q = &port->monitorQueue;
  struct timeval t0,t1;

  ErrnoAbort( pthread_once(&once_block, modeInit) );
  
  qprintf(q,"Running self test on %s\r\n",port->path);

  if (port->isatty)
    {
      long statusDelay = (port->isahub ? 200000 : 100)*1000;
      modemStatus = TIOCM_DTR;
      ioctl(port->fd,TIOCMBIC,&modemStatus);
      nsleep(0,statusDelay);
      ioctl(port->fd,TIOCMGET,&modemStatus);
      dtrCdTestFail  |= (modemStatus & TIOCM_CD) != 0;
      dtrDsrTestFail |= (modemStatus & TIOCM_DSR) != 0;
      
      nsleep(1,0);
        
      modemStatus = TIOCM_DTR;
      ioctl(port->fd,TIOCMBIS,&modemStatus);
      nsleep(0,statusDelay);
      ioctl(port->fd,TIOCMGET,&modemStatus);
      dtrCdTestFail  |= (modemStatus & TIOCM_CD) == 0;
      dtrDsrTestFail |= (modemStatus & TIOCM_DSR) == 0;

      qprintf(q,"DTR->CD  loopback: %s\r\n",  dtrCdTestFail ? "FAIL" : "pass");
      qprintf(q,"DTR->DSR loopback: %s\r\n", dtrDsrTestFail ? "FAIL" : "pass");

      nsleep(1,0);
      
      modemStatus = TIOCM_RTS;
      ioctl(port->fd,TIOCMBIC,&modemStatus);
      nsleep(0,statusDelay);
      ioctl(port->fd,TIOCMGET,&modemStatus);
      rtsRiTestFail  |= (modemStatus & TIOCM_RI) != 0;
      rtsCtsTestFail |= (modemStatus & TIOCM_CTS) != 0;

      nsleep(1,0);
          
      modemStatus = TIOCM_RTS;
      ioctl(port->fd,TIOCMBIS,&modemStatus);
      nsleep(0,statusDelay);
      ioctl(port->fd,TIOCMGET,&modemStatus);
      rtsRiTestFail  |= (modemStatus & TIOCM_RI) == 0;
      rtsCtsTestFail |= (modemStatus & TIOCM_CTS) == 0;
      
      qprintf(q,"RTS->RI  loopback: %s\r\n",  rtsRiTestFail ? "FAIL" : "pass");
      qprintf(q,"RTS->CTS loopback: %s\r\n", rtsCtsTestFail ? "FAIL" : "pass");
    }
  
  qprintf(q,"Data loopback:     ");
  
  gettimeofday(&t0,NULL);

  count = sizeof testPattern;
  while (count)
    {
      i = write(port->fd,testPattern+(sizeof testPattern)-count,count);
      if (i<0)
        {
          qprintf(q,"FAIL -- error writing data\r\n");
          pthread_exit(NULL);
        }
      port->txCount += i;
      count -= i;
    }
  
  pfd.fd = port->fd;
  pfd.events = POLLIN;
  count = sizeof testPattern;
  while (count)
    {
      i = poll(&pfd,1,1000);
      if (i==0)
        {
          qprintf(q,"FAIL -- read data timeout\r\n");
          pthread_exit(NULL);
        }
      if (i<0)
        {
          qprintf(q,"FAIL-- poll() error\r\n");
          pthread_exit(NULL);
        }
      i = read(port->fd,buf+(sizeof testPattern)-count,count);
      if (i<0)
        {
          qprintf(q,"FAIL -- read() error\r\n");
          pthread_exit(NULL);
        }
      port->rxCount += i;
      count -= i;
    }
  
  gettimeofday(&t1,NULL);
  
  delta = timeDeltaMS(&t1,&t0);

  if (memcmp(buf,testPattern,sizeof buf))
    qprintf(q,"FAIL -- data mismatch\r\n");
  else
    qprintf(q,"pass -- %ld bytes in %ldms (%0.1f bytes/sec)\r\n",sizeof buf,delta,
            ((float)(sizeof testPattern)/(float)delta)*1000.0);

  pthread_exit(NULL);
}


void *modeThroughputThread(void * param)
{
  tPortData *port = param;
  int written = 0;
  long txCount = 0;
  long rxCount = 0;
  struct pollfd pfd;
  unsigned char readbuf[256*8];

  fcntl(port->fd, F_SETFL, FNDELAY);
  
  ErrnoAbort( pthread_once(&once_block, modeInit) );
  
  pfd.fd = port->fd;
  pfd.events = POLLIN | POLLOUT;

  while (1)
    {
      int s;
      if (pfd.events == 0)
        {
          Log(1,"port %s pfd.events==0. thread exiting.\n",port->path);
          pthread_exit(NULL);
        }
      s = poll(&pfd,1,100);
      Log(8,"port %s poll() => %d, revents=0x%04x\n",port->path,s,pfd.revents);
      if (s<0 && errno!=EINTR)
        ErrnoAbort(s<0);
      pthread_testcancel();
      while (port->suspend)
        {
          nsleep(0,100000000);
          pthread_testcancel();
        }
      if (s>0)
        {
          if (pfd.revents & POLLERR)
            Log(8,"port %s POLLERR\n",port->path);
          if (pfd.revents & POLLHUP)
            Log(8,"port %s POLLHUP\n",port->path);
          if (pfd.revents & POLLNVAL)
            Log(8,"port %s POLLNVAL\n",port->path);
          if (!(pfd.revents & (POLLIN|POLLOUT)))
              nsleep(0,100000000);
          if (pfd.revents & POLLOUT)
            {
              Log(8,"port %s POLLOUT\n",port->path);
              int w = write(port->fd,testPattern+written,(sizeof testPattern)-written);
              Log(8,"port %s write() => %d\n", port->path, w);
              if (w<0)
                {
                  char buf[256];
                  port->werror = 1;
                  pfd.events &= ~POLLOUT;
                  strerror_r(errno,buf,sizeof buf);
                  Log(1,"port %s write error %d '%s'\n",port->path,errno,buf);
                }
              else
                {
                  txCount += w;
                  written += w;
                  if (written == (sizeof testPattern))
                    written = 0;
                }
            }
        }
      // try doing a read on timeout or when indicated
      if (s==0 || pfd.revents & POLLIN)
        {
          if (s==0)
            Log(1,"port %s poll() timeout\n",port->path);
          if (pfd.revents & POLLIN)
            Log(8,"port %s POLLIN\n",port->path);
          int r = read(port->fd,readbuf,sizeof readbuf);
          Log(8,"port %s read() => %d\n", port->path, r);
          if (r<0 && errno != EAGAIN)
            {
              char buf[256];
              port->rerror = 1;
              pfd.events &= ~POLLIN;
              strerror_r(errno,buf,sizeof buf);
              Log(1,"port %s read error %d '%s'\n",port->path,errno,buf);
            }
          else if (r==0)
            {
              port->eof = 1;
              pfd.events &= ~POLLIN;
              Log(1,"port %s EOF\n",port->path);
            }
          else if (r > 0)
            {
              queueWrite(&port->monitorQueue,readbuf,r);
              rxCount += r;
              if (port->verifyData)
                {
                  int i,j;
                  for (i=0; i<r; ++i)
                    {
                      if (readbuf[i] != testPattern[port->vcount])
                        {
                          int start = i-10;
                          if (start<0)
                            start = 0;
                          port->verror = 1;
                          Log(1,"port %s verify error at %d (read size=%d). thread exiting.\n",port->path,i,r);
                          LogLock(1);
                          LogPrintf(1," rx:");
                          for (j=start; j<r; ++j)
                            LogPrintf(1," %s%02x%s", i==j ? "*" : "", readbuf[j], i==j ? "*" : "");
                          LogPrintf(1,"\n" "exp:");
                          for (j=start; j<r; ++j)
                            LogPrintf(1," %s%02x%s", i==j ? "*" : "", testPattern[((port->vcount-i)+j)%(sizeof testPattern)],  i==j ? "*" : "");
                          LogPrintf(1,"\n");
                          LogUnlock(1);
                          pthread_exit(NULL);
                        }
                      ++port->vcount;
                      if (port->vcount >= (int)(sizeof testPattern))
                        port->vcount = 0;
                    }
                }
            }

          port->rxCount = rxCount;
          port->txCount = txCount;
        }
    }
}


static void *monitor(void * param, int doEcho)
{
  tPortData *port = param;
  long rxCount = 0;
  struct pollfd pfd;
  unsigned char readbuf[256];

  ErrnoAbort( pthread_once(&once_block, modeInit) );
  
  pfd.fd = port->fd;
  pfd.events = POLLIN;

  while (1)
    {
      int s;
      s = poll(&pfd,1,100);
      if (s<0 && errno!=EINTR)
        ErrnoAbort(s<0);
      pthread_testcancel();
      while (port->suspend)
        {
          nsleep(0,100000000);
          pthread_testcancel();
        }
      if (pfd.revents & POLLIN)
        Log(8,"port %s POLLIN\n",port->path);
      if (pfd.revents & POLLERR)
        Log(8,"port %s POLLERR\n",port->path);
      if (pfd.revents & POLLHUP)
        Log(8,"port %s POLLHUP\n",port->path);
      if (pfd.revents & POLLNVAL)
        Log(8,"port %s POLLNVAL\n",port->path);
      if (!(pfd.revents & POLLIN))
        nsleep(0,100000000);
      if (s>0)
        {
          if (pfd.revents & POLLIN)
            {
              int r = read(port->fd,readbuf,sizeof readbuf);
              if (r<0)
                {
                  char buf[256];
                  port->rerror = 1;
                  strerror_r(errno,buf,sizeof buf);
                  Log(1,"port %s read error %d '%s'. sleeping 1 second.\n",port->path,errno,buf);
                  nsleep(1,0);
                  //pthread_exit(NULL);
                }
              else if (r==0)
                {
                  port->eof = 1;
                  Log(1,"port %s EOF. sleeping 1 second.\n",port->path);
                  nsleep(1,0);
                  //pthread_exit(NULL);
                }
              if (doEcho)
                {
                  int w = write(port->fd,readbuf,r);
                  if (w < 0)
                    port->werror = 1;
                  else
                    port->txCount += w;
                }
              queueWrite(&port->monitorQueue,readbuf,r);
              rxCount += r;
            }
          port->rxCount = rxCount;
        }
    }
}



void *modeEchoThread(void *param)
{
  return monitor(param,1);
}

void *modeMonitorThread(void *param)
{
  return monitor(param,0);
}
