#include <cyg/kernel/kapi.h>
#include <cyg/error/codes.h>
#include <cyg/io/io.h>
#include <cyg/io/serialio.h>
#include <cyg/hal/iic.h>
#include <stdio.h>
#include <stdlib.h>

#define InterfaceSelect232 (0<<3)
#define InterfaceSelect422 (1<<3)
#define InterfaceSelect485 (2<<3)
#define InterfaceSelectNone (3<<3)


// Stuff to implement atomic diagnostic message output

// printf routine that prints messages to KS32C5000 UART
extern void diag_printf(const char *fmt, ...);

// atomic diag_printf operation -- only use in running tasks,
// not in initialization code, DSR or ISR code.

#define UseDiagPrintfMutex 1

#if UseDiagPrintfMutex
static cyg_mutex_t dpMutex;
#define diag_printf_m(fmt, args...) \
  do { \
       cyg_mutex_lock(&dpMutex); \
       diag_printf(fmt, ##args); \
       cyg_mutex_unlock(&dpMutex); \
     } while (0)
#define diag_init()  cyg_mutex_init(&dpMutex)
#define diag_lock()  cyg_mutex_lock(&dpMutex)
#define diag_unlock() cyg_mutex_unlock(&dpMutex)
#else
#define diag_printf_m(fmt, args...) diag_printf(fmt, ##args)
#define diag_init()  /* noop */
#define diag_lock()  /* noop */
#define diag_unlock() /* noop */
#endif


typedef unsigned char tStack[4096];

//#define NumPorts 16
unsigned long modelId;
unsigned char NumPorts;
#define Max_Ports 32

//cyg_thread echoThread[NumPorts], backgroundThread;
//tStack echoStack[NumPorts], backgroundStack;
//cyg_handle_t echoHandle[NumPorts], backgroundHandle;
cyg_thread echoThread[Max_Ports], backgroundThread;
tStack echoStack[Max_Ports], backgroundStack;
cyg_handle_t echoHandle[Max_Ports], backgroundHandle;
cyg_thread_entry_t echoTask, backgroundTask;


// Here is where user execution starts

void cyg_user_start(void)
{
  int i;
  diag_printf("Entering cyg_user_start() function\n");

  diag_init();

  // read id stuff from flash
  getModelNumber(&modelId);

  switch (modelId)
    {
    case 5002110:		 //  DM 8 port DB9
    case 5002112:		 //  DM 8 port RJ45
    case 5002115:		 //  DM SH 8 port DB9
    case 5002330:		 //  DM Pro 8 port DB9
      NumPorts = 8;
      break;
    case 5002111:		 //  DM 4 port DB9
    case 5002113:		 //  DM 4 port RJ45
    case 5002116:		 //  IAD UP 4 Port
    case 5002117:                //DM500 4 Port DB9
      NumPorts = 4;
      break;
    case 5002120:		 //  DM 16 port RJ45
    case 5002240:                //  DM RTS RM 16 port
    case 5002315:		 //  DM SH 16 port DB9
    case 5002325:		 //  DM Pro 16 port RJ45
      NumPorts = 16;
      break;
    case 5002241:                //  DM RTS RM 32
      NumPorts = 32;
      break;
    case 5002280:                //  DM RTS 1 Port
    case 5002281:                //  DM 1 Port Embedded
    case 5002283:                //  DM Air 1 Port
    case 5002282:		 //  IAD UP 1 Port
    case 5002284:		 //  IAD UP 1 Port Embedded
    case 5002360:		 //  IAD UP 1 Port 5 - 30 VDC
    case 5002361:		 //  IAD UP 1 Port 5 - 30 VDC Embedded
      NumPorts = 1;
      break;
    case 5002525:                //  DM RTS 2P (Configuration 1)
    case 5002535:                //  DM RTS 2P (Configuration 2)
      NumPorts = 2;
      break;
    default:
      NumPorts = 1;
    }


  for (i=0; i<NumPorts; ++i)
    {
//      setInterfaceType(i,InterfaceSelect232);
      cyg_thread_create(4, echoTask, i,
                        "echo thread", echoStack[i], sizeof echoStack[i],
                        &echoHandle[i],&echoThread[i]);
      cyg_thread_resume(echoHandle[i]);
    }

  cyg_thread_create(5, backgroundTask, (cyg_addrword_t) -1,
                    "background thread", backgroundStack, sizeof backgroundStack,
                    &backgroundHandle,&backgroundThread);
  cyg_thread_resume(backgroundHandle);

  // returning from this function starts scheduler
}


void done(void)
{
  for (;;)
    ;
}

/* per-port count of bytes echoed */

//unsigned totalBytes[NumPorts];
unsigned totalBytes[Max_Ports];

/* this is a simple thread that echos data on a serial port */

void echoTask(cyg_addrword_t data)
{
  unsigned char buf[512];
  int portNum = (int)data;
  cyg_uint32 len;
  Cyg_ErrNo status;
  cyg_io_handle_t ioHandle;
  cyg_serial_info_t serConfig;
  char devName[32];

  diag_printf_m("%d: Beginning execution\n",portNum);

  sprintf(devName,"/dev/ser%d",portNum);

  status = cyg_io_lookup(devName, &ioHandle);

  if (status != ENOERR)
    {
      diag_printf_m("%d: ERROR, cyg_io_lookup returned %d: %s\n",portNum,status,strerror(status));
      done();
    }

  len = sizeof serConfig;
  status = cyg_io_get_config(ioHandle, CYG_IO_GET_CONFIG_SERIAL_INFO, &serConfig, &len);

  if (status != ENOERR)
    {
      diag_printf_m("%d: ERROR, cyg_io_get_config returned %d: %s\n",portNum,status,strerror(status));
      done();
    }

  serConfig.flags &= ~CYG_SERIAL_FLAGS_RTSCTS;
  serConfig.flags &= ~CYG_SERIAL_FLAGS_XONXOFF_RXFLOW;
  serConfig.flags &= ~CYG_SERIAL_FLAGS_XONXOFF_TXFLOW;

  serConfig.flags = CYG_SERIAL_FLAGS_RD_SEMIBLOCK;

  serConfig.baud = 38400;
  serConfig.stop = CYGNUM_SERIAL_STOP_1;
  serConfig.parity = CYGNUM_SERIAL_PARITY_NONE;
  serConfig.word_length = CYGNUM_SERIAL_WORD_LENGTH_8;
  serConfig.interface_mode = InterfaceSelect232;

  len = sizeof serConfig;
  status = cyg_io_set_config(ioHandle, CYG_IO_SET_CONFIG_SERIAL_INFO, &serConfig, &len);

  if (status != ENOERR)
    {
      diag_printf_m("%d: ERROR, cyg_io_get_config returned %d: %s\n",portNum,status,strerror(status));
      done();
    }

  for (;;)
    {
      len = sizeof buf;
      status = cyg_io_read(ioHandle, buf, &len);

      // diag_printf_m("%d[%d]\n",portNum,len);

      // semi-blocking read shouldn't return with len of 0,
      // but let's check just the same

      if (len == 0)
        continue;

      totalBytes[data] += len;

      if (status != ENOERR && status != -EAGAIN)
        {
          diag_printf("%d: ERROR, cyg_io_read returned %d: %s\n",portNum,status,strerror(status));
          done();
        }

      status = cyg_io_write(ioHandle, buf, &len);

      if (status != ENOERR)
        {
          diag_printf("%d: ERROR, cyg_io_write returned %d: %s\n",portNum,status,strerror(status));
          done();
        }
    }
}


// idle thread loop count provided by eCos
extern volatile unsigned idle_thread_loops;

void backgroundTask(cyg_addrword_t data)
{
  int i;
  unsigned lastIdleThreadLoops, thisIdleThreadLoops;

  lastIdleThreadLoops = idle_thread_loops;

  while (1)
    {
      cyg_thread_delay(100);  // 1 second
      thisIdleThreadLoops = idle_thread_loops;

      diag_printf("idle thread loops = %d\n",idle_thread_loops);

      diag_lock();
      diag_printf("Bytes transfered: ");
      for (i=0; i<NumPorts; ++i)
        diag_printf("%d ",totalBytes[i]);
      diag_printf(" -- Idle Loops/Sec: %d\n",thisIdleThreadLoops - lastIdleThreadLoops);
      diag_unlock();

      lastIdleThreadLoops = thisIdleThreadLoops;
    }
}
