#include <cyg/kernel/kapi.h>
#include <cyg/error/codes.h>
#include <cyg/io/io.h>
#include <net/etherio.h>
#include <stdio.h>
#include <stdlib.h>
#include <network.h>


// atomic diag_printf operation -- only use in running tasks,
// not in initialization code, DSR or ISR code.

#define UseDiagPrintfMutex 1

#if UseDiagPrintfMutex
static cyg_mutex_t dpMutex;
#define diag_printf_m(fmt, args...) \
  do { \
       cyg_mutex_lock(&dpMutex); \
       diag_printf(fmt, ##args); \
       cyg_mutex_unlock(&dpMutex); \
     } while (0)
#define diag_init()  cyg_mutex_init(&dpMutex)
#define diag_lock()  cyg_mutex_lock(&dpMutex)
#define diag_unlock() cyg_mutex_unlock(&dpMutex)
#else
#define diag_printf_m(fmt, args...) diag_printf(fmt, ##args)
#define diag_init()  /* noop */
#define diag_lock()  /* noop */
#define diag_unlock() /* noop */
#endif


typedef unsigned char tStack[4096];

cyg_thread   rawEchoThread, backgroundThread;
tStack       rawEchoStack, backgroundStack;
cyg_handle_t rawEchoHandle, backgroundHandle;
cyg_thread_entry_t rawEchoTask, backgroundTask;

// Here is where user execution starts

void cyg_user_start(void)
{
  diag_printf("Entering cyg_user_start() function\n");

  diag_init();

  cyg_thread_create(5, rawEchoTask, (cyg_addrword_t) -1,
                    "raw echo thread", rawEchoStack, sizeof rawEchoStack,
                    &rawEchoHandle,&rawEchoThread);
  cyg_thread_resume(rawEchoHandle);

  cyg_thread_create(6, backgroundTask, (cyg_addrword_t) -1,
                    "background thread", backgroundStack, sizeof backgroundStack,
                    &backgroundHandle,&backgroundThread);
  cyg_thread_resume(backgroundHandle);

  // returning from this function starts scheduler
}

void done(void)
{
  for (;;)
    ;
}

/* count of bytes echoed */

unsigned totalBytes;

#define UseCygIoReadWrite 0

#define protoNumber 0x5432

// this is a simple thread that echos data on a raw Ethernet
// connection using the cyg_io_read() and cyg_io_write() calls

void rawEchoTask(cyg_addrword_t data)
{
  Cyg_ErrNo status;
  cyg_io_handle_t ethHandle;
  cyg_ether_info_t ethConfig;
  cyg_uint32 len;
  unsigned short netorderProto = ntohs(protoNumber);

  diag_printf_m("Beginning execution\n");

  status = cyg_io_lookup("/dev/reth0", &ethHandle);

  if (status != ENOERR)
    {
      diag_printf_m("ERROR, cyg_io_lookup returned %d: %s\n",status,strerror(status));
      done();
    }

  len = sizeof ethConfig;
  status = cyg_io_get_config(ethHandle, CYG_IO_GET_CONFIG_ETHER_INFO, &ethConfig, &len);

  if (status != ENOERR)
    {
      diag_printf_m("ERROR, cyg_io_get_config returned %d: %s\n",status,strerror(status));
      done();
    }

  diag_printf_m("MAC addr %02x:%02x:%02x:%02x:%02x:%02x\n",
                ethConfig.mac_address[0],
                ethConfig.mac_address[1],
                ethConfig.mac_address[2],
                ethConfig.mac_address[3],
                ethConfig.mac_address[4],
                ethConfig.mac_address[5]);

  ethConfig.protocol = netorderProto;

  len = sizeof ethConfig;
  status = cyg_io_set_config(ethHandle, CYG_IO_SET_CONFIG_ETHER_INFO, &ethConfig, &len);

  if (status != ENOERR)
    {
      diag_printf_m("ERROR, cyg_io_set_config returned %d: %s\n",status,strerror(status));
      done();
    }

#if UseCygIoReadWrite

  for (;;)
    {
      static unsigned char rxBuf[2048];
      static unsigned char txBuf[2048];

      len = sizeof rxBuf;
      status = cyg_io_read(ethHandle, rxBuf, &len);

      diag_printf_m("rx [%d]\n",len);

      // blocking read shouldn't return with len of 0, but
      // let's check just the same

      if (len == 0)
        continue;

      totalBytes += len;

      if (status != ENOERR)
        {
          diag_printf("ERROR, cyg_io_read returned %d: %s\n",status,strerror(status));
          done();
        }

      // swap ethernet addresses around, and copy data to tx buffer

      memcpy(txBuf+0,  rxBuf+6, 6);                // dst addr
      memcpy(txBuf+6,  ethConfig.mac_address, 6);  // src addr (ours)
      memcpy(txBuf+12, &netorderProto, 2);         // proto
      memcpy(txBuf+14, rxBuf+14, len-14);          // data


      status = cyg_io_write(ethHandle, txBuf, &len);

      if (status != ENOERR)
        {
          diag_printf("ERROR, cyg_io_write returned %d: %s\n",status,strerror(status));
          done();
        }

      diag_printf_m("tx [%d]\n",len);
    }

#else

#define allocPacket(h,b) cyg_io_get_config(h,CYG_IO_GET_CONFIG_ETHER_ALLOC_BUFFER,b,NULL)
#define getRxPacket(h,b) cyg_io_get_config(h,CYG_IO_GET_CONFIG_ETHER_RECV_BUFFER,b,NULL)
#define freePacket(h,b) cyg_io_get_config(h,CYG_IO_GET_CONFIG_ETHER_FREE_BUFFER,b,NULL)
#define sendPacket(h,b) cyg_io_get_config(h,CYG_IO_GET_CONFIG_ETHER_SEND_BUFFER,b,NULL)

  for (;;)
    {
      tEthBuffer *frame;
      int i;

      status = getRxPacket(ethHandle,&frame);

      if (status != ENOERR)
        {
          diag_printf("ERROR, get packet returned %d: %s\n",status,strerror(status));
          done();
        }

      if (frame->length < 14)
        continue;

      diag_printf_m("rx [%d]\n",frame->length);

      totalBytes += frame->length;

      // Reuse buffer to send.  Remember that receive frames
      // are offset 2 bytes, so we have to shift data.

      memcpy(frame->data+0, frame->data+8, 6);           // dst addr
      memcpy(frame->data+6, ethConfig.mac_address, 6);   // src addr (ours)
      memcpy(frame->data+12, &netorderProto, 2);         // proto
      // don't want to depend on overlapping memcpy() for moving data
      for (i=14; i<frame->length; ++i)
        frame->data[i] = frame->data[i+2];

      status = sendPacket(ethHandle,frame);

      if (status != ENOERR)
        {
          diag_printf("ERROR, sendPacket returned %d: %s\n",status,strerror(status));
          done();
        }

      diag_printf_m("tx [%d]\n",frame->length);
    }
#endif
}


// idle thread loop count provided by eCos
extern volatile unsigned idle_thread_loops;

void backgroundTask(cyg_addrword_t data)
{
  unsigned lastIdleThreadLoops, thisIdleThreadLoops;

  lastIdleThreadLoops = idle_thread_loops;

  while (1)
    {
      cyg_thread_delay(100);  // 1 second
      thisIdleThreadLoops = idle_thread_loops;

      diag_lock();
      diag_printf("Bytes transfered: %d ",totalBytes);
      diag_printf(" -- Idle Loops/Sec: %d\n",thisIdleThreadLoops - lastIdleThreadLoops);
      diag_unlock();

      lastIdleThreadLoops = thisIdleThreadLoops;
    }
}
